import { Header } from "@/components/Header";
import { Footer } from "@/components/Footer";
import { Button } from "@/components/ui/button";
import { Phone, Bed, Users, Wifi, AirVent, Tv, Bath, Coffee, Check } from "lucide-react";
import roomStandard from "@/assets/room-standard.jpg";
import roomDeluxe from "@/assets/room-deluxe.jpg";
import roomPremium from "@/assets/room-premium.jpg";
import heroHotel from "@/assets/hero-hotel.jpg";

const rooms = [
  {
    name: "Standard Room",
    price: "₦12,000",
    image: roomStandard,
    description: "Perfect for solo travelers or couples seeking comfortable accommodation at an affordable price.",
    size: "18 sqm",
    bed: "1 Double Bed",
    guests: "2 Guests",
    features: [
      "Air Conditioning",
      "Flat Screen TV",
      "Free WiFi",
      "Private Bathroom",
      "Daily Housekeeping",
      "24-Hour Power Supply",
    ],
  },
  {
    name: "Deluxe Room",
    price: "₦18,000",
    image: roomDeluxe,
    description: "Spacious and elegantly furnished room with premium amenities for a superior stay experience.",
    size: "24 sqm",
    bed: "1 King Bed",
    guests: "2 Guests",
    popular: true,
    features: [
      "Air Conditioning",
      "Flat Screen TV",
      "Free WiFi",
      "Private Bathroom",
      "Mini Refrigerator",
      "Work Desk",
      "Room Service",
      "Daily Housekeeping",
    ],
  },
  {
    name: "Premium Suite",
    price: "₦25,000",
    image: roomPremium,
    description: "Our finest accommodation featuring a separate living area, luxurious furnishings, and exclusive amenities.",
    size: "35 sqm",
    bed: "1 King Bed + Sofa",
    guests: "3 Guests",
    features: [
      "Air Conditioning",
      "Flat Screen TV",
      "Free WiFi",
      "Luxury Bathroom",
      "Mini Refrigerator",
      "Living Area",
      "Work Desk",
      "Coffee/Tea Maker",
      "Complimentary Breakfast",
      "Priority Room Service",
    ],
  },
];

const Rooms = () => {
  return (
    <div className="min-h-screen">
      <Header />
      <main>
        {/* Hero Banner */}
        <section className="relative h-[40vh] min-h-[300px] flex items-center justify-center">
          <div
            className="absolute inset-0 bg-cover bg-center"
            style={{ backgroundImage: `url(${heroHotel})` }}
          >
            <div className="absolute inset-0 bg-emerald-dark/70" />
          </div>
          <div className="relative z-10 text-center px-4">
            <span className="text-gold font-medium tracking-widest uppercase text-sm">
              Accommodations
            </span>
            <h1 className="font-serif text-4xl md:text-5xl lg:text-6xl font-bold text-primary-foreground mt-2">
              Our Rooms
            </h1>
            <p className="text-primary-foreground/80 mt-4 max-w-xl mx-auto">
              Choose from our selection of comfortable and well-appointed rooms
            </p>
          </div>
        </section>

        {/* Rooms List */}
        <section className="py-16 md:py-20 bg-background">
          <div className="container mx-auto px-4">
            <div className="space-y-16 max-w-6xl mx-auto">
              {rooms.map((room, index) => (
                <div
                  key={room.name}
                  className={`grid lg:grid-cols-2 gap-8 items-center ${
                    index % 2 === 1 ? "lg:flex-row-reverse" : ""
                  }`}
                >
                  <div className={index % 2 === 1 ? "lg:order-2" : ""}>
                    <div className="relative">
                      <img
                        src={room.image}
                        alt={room.name}
                        className="rounded-2xl shadow-elegant w-full h-[300px] md:h-[400px] object-cover"
                      />
                      {room.popular && (
                        <span className="absolute top-4 left-4 bg-gold text-emerald-dark text-xs font-bold px-3 py-1 rounded-full uppercase">
                          Most Popular
                        </span>
                      )}
                    </div>
                  </div>
                  <div className={index % 2 === 1 ? "lg:order-1" : ""}>
                    <h2 className="font-serif text-3xl md:text-4xl font-bold text-foreground mb-2">
                      {room.name}
                    </h2>
                    <p className="text-muted-foreground mb-6">{room.description}</p>
                    
                    <div className="flex flex-wrap gap-4 mb-6">
                      <div className="flex items-center gap-2 text-sm text-muted-foreground">
                        <Bed className="w-4 h-4 text-gold" />
                        {room.bed}
                      </div>
                      <div className="flex items-center gap-2 text-sm text-muted-foreground">
                        <Users className="w-4 h-4 text-gold" />
                        {room.guests}
                      </div>
                      <div className="flex items-center gap-2 text-sm text-muted-foreground">
                        <span className="w-4 h-4 text-gold font-bold text-xs">m²</span>
                        {room.size}
                      </div>
                    </div>

                    <div className="grid grid-cols-2 gap-2 mb-6">
                      {room.features.map((feature) => (
                        <div key={feature} className="flex items-center gap-2 text-sm text-muted-foreground">
                          <Check className="w-4 h-4 text-primary" />
                          {feature}
                        </div>
                      ))}
                    </div>

                    <div className="flex items-center gap-4 mb-6">
                      <span className="text-3xl font-bold text-gold">{room.price}</span>
                      <span className="text-muted-foreground">/night</span>
                    </div>

                    <a href="tel:07045064559">
                      <Button size="lg" className="gap-2">
                        <Phone className="w-5 h-5" />
                        Book This Room
                      </Button>
                    </a>
                  </div>
                </div>
              ))}
            </div>
          </div>
        </section>

        {/* Policies */}
        <section className="py-16 bg-secondary">
          <div className="container mx-auto px-4">
            <div className="max-w-4xl mx-auto">
              <h2 className="font-serif text-3xl font-bold text-foreground text-center mb-8">
                Room Policies
              </h2>
              <div className="grid md:grid-cols-3 gap-6">
                <div className="bg-card p-6 rounded-xl">
                  <h3 className="font-semibold text-foreground mb-2">Check-in Time</h3>
                  <p className="text-muted-foreground">From 2:00 PM onwards</p>
                </div>
                <div className="bg-card p-6 rounded-xl">
                  <h3 className="font-semibold text-foreground mb-2">Check-out Time</h3>
                  <p className="text-muted-foreground">By 12:00 PM (Noon)</p>
                </div>
                <div className="bg-card p-6 rounded-xl">
                  <h3 className="font-semibold text-foreground mb-2">Cancellation</h3>
                  <p className="text-muted-foreground">Free cancellation up to 24 hours before check-in</p>
                </div>
              </div>
            </div>
          </div>
        </section>

        {/* CTA */}
        <section className="py-16 bg-primary">
          <div className="container mx-auto px-4 text-center">
            <h2 className="font-serif text-3xl md:text-4xl font-bold text-primary-foreground mb-4">
              Need Help Choosing?
            </h2>
            <p className="text-primary-foreground/80 mb-8 max-w-xl mx-auto">
              Our friendly staff is ready to help you find the perfect room for your stay.
            </p>
            <a href="tel:07045064559">
              <Button size="lg" variant="secondary" className="gap-2">
                <Phone className="w-5 h-5" />
                Call Us Now
              </Button>
            </a>
          </div>
        </section>
      </main>
      <Footer />
    </div>
  );
};

export default Rooms;
