import { useState } from "react";
import { Header } from "@/components/Header";
import { Footer } from "@/components/Footer";
import { Button } from "@/components/ui/button";
import { X, Phone } from "lucide-react";
import heroHotel from "@/assets/hero-hotel.jpg";
import roomStandard from "@/assets/room-standard.jpg";
import roomDeluxe from "@/assets/room-deluxe.jpg";
import roomPremium from "@/assets/room-premium.jpg";
import galleryReception from "@/assets/gallery-reception.jpg";
import galleryRestaurant from "@/assets/gallery-restaurant.jpg";

const categories = ["All", "Exterior", "Rooms", "Facilities"];

const galleryImages = [
  { src: heroHotel, alt: "Hotel Exterior View", category: "Exterior" },
  { src: galleryReception, alt: "Hotel Reception & Lobby", category: "Facilities" },
  { src: roomStandard, alt: "Standard Room", category: "Rooms" },
  { src: roomDeluxe, alt: "Deluxe Room", category: "Rooms" },
  { src: roomPremium, alt: "Premium Suite", category: "Rooms" },
  { src: galleryRestaurant, alt: "Restaurant & Dining", category: "Facilities" },
];

const Gallery = () => {
  const [selectedImage, setSelectedImage] = useState<string | null>(null);
  const [activeCategory, setActiveCategory] = useState("All");

  const filteredImages = activeCategory === "All" 
    ? galleryImages 
    : galleryImages.filter(img => img.category === activeCategory);

  return (
    <div className="min-h-screen">
      <Header />
      <main>
        {/* Hero Banner */}
        <section className="relative h-[40vh] min-h-[300px] flex items-center justify-center">
          <div
            className="absolute inset-0 bg-cover bg-center"
            style={{ backgroundImage: `url(${heroHotel})` }}
          >
            <div className="absolute inset-0 bg-emerald-dark/70" />
          </div>
          <div className="relative z-10 text-center px-4">
            <span className="text-gold font-medium tracking-widest uppercase text-sm">
              Visual Tour
            </span>
            <h1 className="font-serif text-4xl md:text-5xl lg:text-6xl font-bold text-primary-foreground mt-2">
              Photo Gallery
            </h1>
            <p className="text-primary-foreground/80 mt-4 max-w-xl mx-auto">
              Explore our beautiful hotel through our collection of photos
            </p>
          </div>
        </section>

        {/* Gallery Section */}
        <section className="py-16 md:py-20 bg-background">
          <div className="container mx-auto px-4">
            {/* Category Filter */}
            <div className="flex flex-wrap justify-center gap-3 mb-12">
              {categories.map((category) => (
                <button
                  key={category}
                  onClick={() => setActiveCategory(category)}
                  className={`px-6 py-2 rounded-full text-sm font-medium transition-all ${
                    activeCategory === category
                      ? "bg-primary text-primary-foreground"
                      : "bg-secondary text-foreground hover:bg-secondary/80"
                  }`}
                >
                  {category}
                </button>
              ))}
            </div>

            {/* Gallery Grid */}
            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-6 max-w-6xl mx-auto">
              {filteredImages.map((image, index) => (
                <div
                  key={index}
                  className="relative overflow-hidden rounded-xl cursor-pointer group aspect-[4/3]"
                  onClick={() => setSelectedImage(image.src)}
                >
                  <img
                    src={image.src}
                    alt={image.alt}
                    className="w-full h-full object-cover transition-transform duration-500 group-hover:scale-110"
                  />
                  <div className="absolute inset-0 bg-gradient-to-t from-emerald-dark/80 via-transparent to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-300">
                    <div className="absolute bottom-4 left-4 right-4">
                      <span className="text-xs text-gold font-medium uppercase tracking-wider">
                        {image.category}
                      </span>
                      <p className="text-primary-foreground font-semibold mt-1">{image.alt}</p>
                    </div>
                  </div>
                </div>
              ))}
            </div>

            {filteredImages.length === 0 && (
              <p className="text-center text-muted-foreground py-12">
                No images found in this category.
              </p>
            )}
          </div>
        </section>

        {/* CTA */}
        <section className="py-16 bg-primary">
          <div className="container mx-auto px-4 text-center">
            <h2 className="font-serif text-3xl md:text-4xl font-bold text-primary-foreground mb-4">
              See It For Yourself
            </h2>
            <p className="text-primary-foreground/80 mb-8 max-w-xl mx-auto">
              Pictures only tell part of the story. Book your stay and experience Graziella in person.
            </p>
            <div className="flex flex-col sm:flex-row gap-4 justify-center">
              <a href="tel:07045064559">
                <Button size="lg" variant="secondary" className="gap-2">
                  <Phone className="w-5 h-5" />
                  Book Your Stay
                </Button>
              </a>
              <a href="/rooms">
                <Button size="lg" variant="heroOutline">
                  View Room Options
                </Button>
              </a>
            </div>
          </div>
        </section>
      </main>
      <Footer />

      {/* Lightbox */}
      {selectedImage && (
        <div
          className="fixed inset-0 z-50 bg-black/90 flex items-center justify-center p-4"
          onClick={() => setSelectedImage(null)}
        >
          <button
            className="absolute top-4 right-4 text-white hover:text-gold transition-colors"
            onClick={() => setSelectedImage(null)}
          >
            <X className="w-8 h-8" />
          </button>
          <img
            src={selectedImage}
            alt="Gallery preview"
            className="max-w-full max-h-[90vh] object-contain rounded-lg"
            onClick={(e) => e.stopPropagation()}
          />
        </div>
      )}
    </div>
  );
};

export default Gallery;
