import { useState, useEffect } from "react";
import { Link, useLocation } from "react-router-dom";
import { Menu, X, Phone } from "lucide-react";
import { Button } from "@/components/ui/button";

const navLinks = [
  { label: "Home", href: "/" },
  { label: "About", href: "/about" },
  { label: "Rooms", href: "/rooms" },
  { label: "Amenities", href: "/amenities" },
  { label: "Gallery", href: "/gallery" },
  { label: "Contact", href: "/contact" },
];

export const Header = () => {
  const [isScrolled, setIsScrolled] = useState(false);
  const [isMobileMenuOpen, setIsMobileMenuOpen] = useState(false);
  const location = useLocation();
  const isHomePage = location.pathname === "/";

  useEffect(() => {
    const handleScroll = () => {
      setIsScrolled(window.scrollY > 50);
    };
    window.addEventListener("scroll", handleScroll);
    return () => window.removeEventListener("scroll", handleScroll);
  }, []);

  const showTransparent = isHomePage && !isScrolled;

  return (
    <header
      className={`fixed top-0 left-0 right-0 z-50 transition-all duration-300 ${
        showTransparent
          ? "bg-transparent"
          : "bg-card/95 backdrop-blur-md shadow-card"
      }`}
    >
      <div className="container mx-auto px-4">
        <div className="flex items-center justify-between h-20">
          {/* Logo */}
          <Link to="/" className="flex flex-col">
            <span
              className={`font-serif text-xl md:text-2xl font-bold transition-colors ${
                showTransparent ? "text-primary-foreground" : "text-primary"
              }`}
            >
              Graziella
            </span>
            <span
              className={`text-xs tracking-widest uppercase transition-colors ${
                showTransparent ? "text-gold-light" : "text-gold"
              }`}
            >
              Hotels & Suites
            </span>
          </Link>

          {/* Desktop Navigation */}
          <nav className="hidden lg:flex items-center gap-8">
            {navLinks.map((link) => (
              <Link
                key={link.href}
                to={link.href}
                className={`text-sm font-medium transition-colors hover:text-gold ${
                  showTransparent ? "text-primary-foreground" : "text-foreground"
                } ${location.pathname === link.href ? "text-gold" : ""}`}
              >
                {link.label}
              </Link>
            ))}
          </nav>

          {/* Call Button - Desktop */}
          <div className="hidden lg:flex items-center gap-4">
            <a href="tel:07045064559">
              <Button variant={showTransparent ? "heroOutline" : "default"} size="sm">
                <Phone className="w-4 h-4" />
                Call Now
              </Button>
            </a>
          </div>

          {/* Mobile Menu Button */}
          <button
            className="lg:hidden p-2"
            onClick={() => setIsMobileMenuOpen(!isMobileMenuOpen)}
            aria-label="Toggle menu"
          >
            {isMobileMenuOpen ? (
              <X className={`w-6 h-6 ${showTransparent ? "text-primary-foreground" : "text-foreground"}`} />
            ) : (
              <Menu className={`w-6 h-6 ${showTransparent ? "text-primary-foreground" : "text-foreground"}`} />
            )}
          </button>
        </div>
      </div>

      {/* Mobile Menu */}
      <div
        className={`lg:hidden absolute top-full left-0 right-0 bg-card shadow-lg transition-all duration-300 ${
          isMobileMenuOpen ? "opacity-100 visible" : "opacity-0 invisible"
        }`}
      >
        <nav className="container mx-auto px-4 py-4 flex flex-col gap-2">
          {navLinks.map((link) => (
            <Link
              key={link.href}
              to={link.href}
              onClick={() => setIsMobileMenuOpen(false)}
              className={`text-foreground font-medium py-3 px-4 rounded-lg hover:bg-secondary transition-colors ${
                location.pathname === link.href ? "bg-secondary text-gold" : ""
              }`}
            >
              {link.label}
            </Link>
          ))}
          <a href="tel:07045064559" className="mt-2">
            <Button className="w-full" size="lg">
              <Phone className="w-4 h-4" />
              Call Now
            </Button>
          </a>
        </nav>
      </div>
    </header>
  );
};
